/* getcursor.c
 *
 * Functions to deal with allocating and freeing cursors for Oracle
 */
/* Copyright 1991, 1992, 1993 Kevin Stock.
 *
 * You may copy this under the terms of the GNU General Public License,
 * or the Artistic License, copies of which should have accompanied your
 * Perl kit.
 */

#include	"EXTERN.h"
#include	<stdio.h>
#include	<ctype.h>
#include	"orafns.h"


/* head of the cursor list */
struct cursor csr_list =
	{ NULL, NULL, NULL, NULL, NULL, NULL, 0, 0, 0, 0, 0, 0, NULL, NULL };


/* ora_free_data(csr)
 *
 * Frees memory attached to csr->data
 */

void ora_free_data(csr)
struct cursor *csr;
{
	int i;

	DBUG_ENTER("ora_free_data");
	DBUG_PRINT("entry", ("ora_free_data(%#lx)", (long) csr));

	if (csr->data == NULL)
	{
		DBUG_PRINT("exit", ("nothing to do"));
		DBUG_VOID_RETURN;
	}

	for (i = 0 ; (i < csr->nfields) && (csr->data[i] != NULL) ; i++)
	{
		DBUG_PRINT("free", ("freeing field %d at address %#lx",
		     i, (long) csr->data[i]));
		free(csr->data[i]);
	}

	DBUG_PRINT("free", ("freeing cursor data at address %#lx",
	    (long) csr->data));
	free(csr->data);
	csr->data = (char **) NULL;

	DBUG_VOID_RETURN;
}


/* ora_free_rcode(csr)
 *
 * Frees memory attached to csr->rcode
 */

void ora_free_rcode(csr)
struct cursor *csr;
{
	int i;

	DBUG_ENTER("ora_free_rcode");
	DBUG_PRINT("entry", ("ora_free_rcode(%#lx)", (long) csr));

	if (csr->rcode == NULL)
	{
		DBUG_PRINT("exit", ("nothing to do"));
		DBUG_VOID_RETURN;
	}

	for (i = 0 ; (i < csr->nfields) && (csr->rcode[i] != NULL) ; i++)
	{
		DBUG_PRINT("free", ("freeing rcode %d at address %#lx",
		     i, (long) csr->rcode[i]));
		free(csr->rcode[i]);
	}

	DBUG_PRINT("free", ("freeing cursor rcode at address %#lx",
	    (long) csr->rcode));
	free(csr->rcode);
	csr->rcode = (short **) NULL;

	DBUG_VOID_RETURN;
}


/* ora_getcursor()
 *
 * Allocates memory for a new cursor and returns its address.
 * Inserts the cursor at the front of the list.
 * Returns NULL if it can't get enough memory.
 */

struct cursor *ora_getcursor()
{
	struct cursor *tmp;

	DBUG_ENTER("ora_getcursor");
	DBUG_PRINT("entry", ("ora_getcursor(void)"));

	if ((tmp = (struct cursor *) malloc(sizeof(struct cursor))) == NULL)
	{
		DBUG_PRINT("malloc", ("insufficient memory for a cursor"));
		DBUG_PRINT("exit", ("returning NULL"));
		ora_err.no = ORAP_NOMEM;
		ora_err.lda = NULL;
		DBUG_RETURN(NULL);
	}
	DBUG_PRINT("malloc", ("got a cursor at %#lx", (long) tmp));

	if ((tmp->csr = (struct csrdef *)malloc(sizeof(struct csrdef))) == NULL)
	{
		free(tmp);
		DBUG_PRINT("malloc", ("insufficient memory for a csrdef"));
		DBUG_PRINT("exit", ("returning NULL"));
		ora_err.no = ORAP_NOMEM;
		ora_err.lda = NULL;
		DBUG_RETURN(NULL);
	}
	DBUG_PRINT("malloc", ("got a csr at %#lx", (long) tmp->csr));

	tmp->hda = NULL;
	tmp->data = NULL;
	tmp->rcode = NULL;
	tmp->len = NULL;
	tmp->type = NULL;
	tmp->cache_size = 0;
	tmp->end_of_data = 0;
	tmp->in_cache = 0;
	tmp->next_entry = 0;
	tmp->nfields = 0;
	tmp->varfields = 0;
	tmp->parent = NULL;

	tmp->next = csr_list.next;
	csr_list.next = tmp;

	ora_err.no = 0;
	ora_err.lda = NULL;
	DBUG_PRINT("exit", ("returning %#lx", (long) tmp));
	DBUG_RETURN(tmp);
}


/* ora_getlda()
 *
 * Gets a new login data area.
 * Uses ora_getcursor and then allocates the host data area.
 */

struct cursor *ora_getlda()
{
	struct cursor *tmp;

	DBUG_ENTER("ora_getlda");
	DBUG_PRINT("entry", ("ora_getlda(void)"));

	if ((tmp = ora_getcursor()) == NULL)
	{
		DBUG_PRINT("exit", ("returning NULL"));
		DBUG_RETURN(NULL);
	}

	if ((tmp->hda = malloc(256)) == NULL)
	{
		DBUG_PRINT("malloc", ("insufficient memory for an hda"));
		ora_dropcursor(tmp);
		ora_err.no = ORAP_NOMEM;
		ora_err.lda = NULL;
		DBUG_PRINT("exit", ("returning NULL"));
		DBUG_RETURN(NULL);
	}
	DBUG_PRINT("malloc", ("got hda at %#lx", tmp->hda));
	DBUG_PRINT("exit", ("returning %#lx", tmp));
	DBUG_RETURN(tmp);
}


/* ora_dropcursor(csr)
 *
 * Frees the space occupied by a given cursor, removing it from the list.
 */

int ora_dropcursor(csr)
struct cursor *csr;
{
	struct cursor *tmp, *t;

	DBUG_ENTER("ora_dropcursor");
	DBUG_PRINT("entry", ("ora_dropcursor(%#lx)", (long) csr));

	tmp = &csr_list;

	while ((tmp->next != NULL) && (tmp->next != csr))
	{
		tmp = tmp->next;
	}

	if (tmp->next == NULL)
	{
		ora_err.no = ORAP_INVCSR;
		ora_err.lda = NULL;
		DBUG_PRINT("exit", ("not a cursor"));
		DBUG_RETURN(0);
	}

	t = tmp->next;

	if (t->csr != NULL)
	{
		DBUG_PRINT("free", ("freeing csr at %#lx", (long) t->csr));
		free(t->csr);
	}
	if (t->hda != NULL)
	{
		DBUG_PRINT("free", ("freeing hda at %#lx", (long) t->hda));
		free(t->hda);
	}
	if (t->data != NULL)
	{
		ora_free_data(t);
	}
	if (t->rcode != NULL)
	{
		ora_free_rcode(t);
	}
	if (t->len != NULL)
	{
		DBUG_PRINT("free", ("freeing len array at %#lx", (long)t->len));
		free(t->len);
	}
	if (t->type != NULL)
	{
		DBUG_PRINT("free",("freeing type array at %#lx",(long)t->type));
		free(t->type);
	}

	t = t->next;
	DBUG_PRINT("free", ("freeing cursor at %#lx", (long) tmp->next));
	free(tmp->next);
	tmp->next = t;

	DBUG_PRINT("exit", ("completed"));
	DBUG_RETURN(1);
}


/* ora_droplda()
 *
 * This is just here for completeness' sake.
 * (I suppose we could check the value of hda in dropcursor and droplda
 * but I don't think it's worth it)
 */

int ora_droplda(lda)
struct cursor *lda;
{
	int ret;

	DBUG_ENTER("ora_droplda");
	DBUG_PRINT("entry", ("ora_droplda(%#lx)", (long) lda));

	ret = ora_dropcursor(lda);

	DBUG_PRINT("exit", ("returning %d", ret));
	DBUG_RETURN(ret);
}


/* ora_findcursor()
 *
 * Checks whether the specified csr is present in the list
 */

int ora_findcursor(csr)
struct cursor *csr;
{
	struct cursor *tmp = &csr_list;

	DBUG_ENTER("ora_findcursor");
	DBUG_PRINT("entry", ("ora_findcursor(%#lx)", (long) csr));

	while ((tmp->next != NULL) && (tmp->next != csr))
	{
		tmp = tmp->next;
	}

	if (tmp->next == NULL)
	{
		DBUG_PRINT("exit", ("not a cursor"));
		DBUG_RETURN(0);
	}

	DBUG_PRINT("exit", ("is a cursor"));
	DBUG_RETURN(1);
}


/* check_lda()
 *
 * Checks whether the given address corresponds to a valid lda
 */

 int check_lda(lda)
 struct cursor *lda;
 {
	DBUG_ENTER("check_lda");
	DBUG_PRINT("entry", ("check_lda(%#lx)", (long) lda));

	if (ora_findcursor(lda) && (lda->hda != NULL) && (lda->data == NULL))
	{
		DBUG_PRINT("exit", ("is an lda"));
		DBUG_RETURN(1);
	}
	else
	{
		DBUG_PRINT("exit", ("not an lda"));
		DBUG_RETURN(0);
	}
};


/* check_csr()
 *
 * Checks whether the given address corresponds to a valid csr
 */

 int check_csr(csr)
 struct cursor *csr;
 {
	DBUG_ENTER("check_csr");
	DBUG_PRINT("entry", ("check_csr(%#lx)", (long) csr));

	if (ora_findcursor(csr) && (csr->hda == NULL))
	{
		DBUG_PRINT("exit", ("is a csr"));
		DBUG_RETURN(1);
	}
	else
	{
		DBUG_PRINT("exit", ("not a csr"));
		DBUG_RETURN(0);
	}
};
